define([
    'underscore',
    'backbone',
    'modules/new-appointment-request/views/common/method/section-view',
    'modules/new-appointment-request/tests/helpers/facility-helper',
], function(_, Backbone, View, helpers) {
    'use strict';


    describe('Method Selection View', function() {

        var region;
        var model;

        describe('VAR-7912 Specification', function() {

            /**
             * Helper function to show the view
             * @param {Boolean} [isRequest=false]
             * @param {Boolean} [isDirect=false]
             * @param {*} [options]
             * @return {Backbone.View<Backbone.Model>}
             */
            function showView(isRequest, isDirect, options) {
                var _isRequest = isRequest || false;
                var _isDirect = isDirect || false;
                var _options = options || {};

                _.extend(_options, {
                    isRequestSupported: _isRequest,
                    isDirectSchedulingSupported: _isDirect,
                    model: model,
                });

                region.show(new View(_options));
                return region.currentView;
            }


            beforeEach(function() {
                region = helpers.setupTestingRegion();
                model = new Backbone.Model();
            });

            /**
             * 4.        Radio button options; unless otherwise noted below, if the option is supported based on the
             *           utility configuration, the option is displayed as enabled
             *
             * 4.1.      Schedule myself
             *
             * 4.2.      Submit a Request to a VA Scheduler
             */
            describe('4.0', function() {

                /**
                 * Helpers function
                 * @param {Marionette.View} view
                 * @param {string} selector
                 * @return {boolean}
                 */
                function isButtonEnabled(view, selector) {
                    var button = view.$(selector)[0];
                    return !_.has(button.attributes, 'disabled');
                }

                function getLabelText(view, selector) {
                    var button = view.$(selector);
                    var label = button.parent();
                    var p = label.find('p');

                    return p[0].innerText.trim();
                }

                /**
                 * 4.1.1.     If the request is for primary care and the veteran has no assigned PACT team, the selection
                 *            is disabled following text below the radio button selection:
                 *
                 * 4.1.1.1.   You must have an assigned PACT team to directly schedule appointments. Please contact your
                 *            facility for assistance.
                 *
                 * 4.1.2.     If the method is not supported display the option as disabled with the following text below
                 *            the radio button selection:
                 *
                 * 4.1.2.1.   This method is not available for the type of care you selected.
                 *
                 * 4.1.2.2.   If the method is not supported due to a required patient history limit, the option is
                 *            disabled and the following text is displayed: This method is not available because you
                 *            have not received this type of care in the last <xx> months. Please select another method,
                 *            if available, or contact your facility for assistance.
                 *
                 * 4.1.2.2.1  Insert 12 or 24 depending on limit
                 */
                describe('Scheduling Myself', function() {
                    var selector = 'input[type="radio"][value="direct"]';

                    beforeEach(function() {
                        model.set('facility', new Backbone.Model());
                    });

                    it('disables direct scheduling for primary care and the users has no PAC team', function() {
                        var view = showView(false, true, {
                            isPacNeeded: true,
                            pacTeam: {
                                hasPacTeamAtFacility: function() {
                                    return false;
                                },
                            },
                        });
                        var isEnabled = isButtonEnabled(view, selector);

                        expect(isEnabled).toBeFalsy();
                    });

                    it('generates the correct message when the PACT team is missing', function() {
                        var options = {
                            isRequestSupported: false,
                            isDirectSchedulingSupported: true,
                            model: model,
                            isPacNeeded: true,
                            pacTeam: {
                                hasPacTeamAtFacility: function() {
                                    return false;
                                },
                            },

                        };
                        var V = View.extend({
                            isClinicsEmpty: function() {
                                return false;
                            },
                        });
                        var view;
                        var expected;
                        var text;

                        region.show(new V(options));

                        view = region.currentView;

                        expected = 'You must have an assigned PACT team to directly schedule appointments. ' +
                            'Please contact your facility for assistance.';
                        text = getLabelText(view, selector);

                        expect(text).toEqual(expected);
                    });

                    it('disables the button when the method is not supported', function() {
                        var view = showView();
                        var isEnabled = isButtonEnabled(view, selector);

                        expect(isEnabled).toBeFalsy();
                    });

                    it('creates the correct message when the method is not supported', function() {
                        var view = showView();

                        var expected = 'This method is not available for the type of care you selected.';
                        var text = getLabelText(view, selector);

                        expect(text).toEqual(expected);
                    });

                    it('disables the button because of a direct history limit', function() {
                        var view = showView(false, true, {
                            isDirectHistoryNeeded: true,
                            directHistory: new (Backbone.Model.extend({
                                hasRecentVisit: function() {
                                    return false;
                                },
                            }))({durationInMonths: 24}),
                        });

                        var isEnabled = isButtonEnabled(view, selector);

                        expect(isEnabled).toBeFalsy();
                    });

                    it('generates the correct message when it is missing the patient direct history limit', function() {
                        var options = {
                            isRequestSupported: false,
                            isDirectSchedulingSupported: true,
                            isDirectHistoryNeeded: true,
                            directHistory: new (Backbone.Model.extend({
                                hasRecentVisit: function() {
                                    return false;
                                },
                            }))({durationInMonths: 24}),
                            model: model,
                        };
                        var V = View.extend({
                            isClinicsEmpty: function() {
                                return false;
                            },
                        });
                        var view;
                        var expected;
                        var text;

                        region.show(new V(options));

                        view = region.currentView;

                        expected = 'This method is not available because you have not received this type of care in' +
                            ' the last 24 months. Please select another method, if available, or contact your facility' +
                            ' for assistance.';

                        text = getLabelText(view, selector);

                        expect(text).toEqual(expected);
                    });
                });


                /**
                 * 4.2.1.     If the request limit has been reached, the option is shown as disabled and the following
                 *            text is shown below the radio button selection:
                 *
                 * 4.2.1.1.   You have reached the limit of <number limit> <type of care> requests. In order to proceed,
                 *            please cancel one of the pending requests, or wait for the VA to process one of the requests.
                 *
                 * 4.2.2.     If the method is not supported display the option as disabled with the following text below
                 *            the radio button selection:
                 *
                 * 4.2.2.1.   This method is not available for the type of care you selected.
                 *
                 * 4.2.2.2.   If the method is not supported due to a required patient history limit, the option is
                 *            disabled and the following text is displayed: This method is not available because you
                 *            have not received this type of care in the last <xx> months. Please select another method,
                 *            if available, or contact your facility for assistance.
                 *
                 * 4.2.2.2.1  Insert 12 or 24 depending on limit
                 */
                describe('Submit a Request to a VA Scheduler', function() {
                    var selector = 'input[type="radio"][value="clerk"]';

                    beforeEach(function() {
                        model.isExpressCare = function() {
                            return false;
                        };
                    });

                    it('disables the request button when the limit is reached', function() {
                        var view = showView(true, false, {
                            isLimitNeeded: true,
                            requestLimit: new (Backbone.Model.extend({
                                hasRequestsLeft: function() {
                                    return false;
                                },
                            }))({requestLimit: 5}),
                        });
                        var isEnabled = isButtonEnabled(view, selector);

                        expect(isEnabled).toBeFalsy();
                    });

                    it('shows the correct message when the request limit is reached', function() {
                        var view = showView(true, false, {
                            isLimitNeeded: true,
                            typeOfCareName: 'Zombie Face Lift',
                            requestLimit: new (Backbone.Model.extend({
                                hasRequestsLeft: function() {
                                    return false;
                                },
                            }))({requestLimit: 5}),
                        });

                        var expected = 'You have reached the limit of 5 Zombie Face Lift request(s).\nIn order to proceed, ' +
                            'please cancel a pending request, or wait for the VA to process a request.';
                        var text = getLabelText(view, selector);

                        expect(text).toEqual(expected);
                    });

                    it('disables the button when the request method is not supported', function() {
                        var view = showView();
                        var isEnabled = isButtonEnabled(view, selector);

                        expect(isEnabled).toBeFalsy();
                    });

                    it('shows the correct message when the request method is not supported', function() {
                        var view = showView();

                        var expected = 'This method is not available for the type of care you selected.';
                        var text = getLabelText(view, selector);

                        expect(text).toEqual(expected);
                    });

                    it('disables the request button when you have not met the patient history limit', function() {
                        var view = showView(true, false, {
                            isRequestHistoryNeeded: true,
                            requestHistory: new (Backbone.Model.extend({
                                hasRecentVisit: function() {
                                    return false;
                                },
                                isExpressCare: function() {
                                    return false;
                                },
                            }))({durationInMonths: 24}),
                        });
                        var isEnabled = isButtonEnabled(view, selector);

                        expect(isEnabled).toBeFalsy();
                    });

                    it('shows the correct message when you have not met the patient history limit', function() {
                        var view = showView(true, false, {
                            isRequestHistoryNeeded: true,
                            requestHistory: new (Backbone.Model.extend({
                                hasRecentVisit: function() {
                                    return false;
                                },
                            }))({durationInMonths: 24}),
                        });
                        var expected = 'This method is not available because you have not received this type of care in ' +
                            'the last 24 months. Please select another method, if available, or contact your facility ' +
                            'for assistance.';
                        var text = getLabelText(view, selector);

                        expect(text).toEqual(expected);
                    });
                });
            });
        });
    });
});
